package com.infoepoch.pms.dispatchassistant.common.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.infoepoch.pms.dispatchassistant.common.exception.ValidationException;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * JsonNode工具类
 */
public class JsonNodeUtils {

    // 定义 ObjectMappe 对象
    private static final ObjectMapper MAPPER = new ObjectMapper();

    /**
     * 解析JsonNode中的String值并去除空格，解析不成功返回默认值
     */
    public static String toString(JsonNode node, String defaultValue) {
        if (node == null || node.isNull() || StringUtils.isBlank(node.asText().trim()))
            return defaultValue;
        return node.asText().trim();
    }

    /**
     * java 对象转换为 json 字符串
     *
     * @param objectData java 对象
     * @return String    json 字符串
     */
    public static String objectToJson(Object objectData) {
        try {
            String string = MAPPER.writeValueAsString(objectData);
            return string;
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * json 字符串转换为 JsonNode
     *
     * @param jsonData json 字符串
     * @return JsonNode
     */
    public static JsonNode jsonToJsonNode(String jsonData) throws IOException {
        JsonNode jsonNode = MAPPER.readTree(jsonData);
        return jsonNode;
    }

    /**
     * 解析JsonNode中的String值并去除空格，解析不成功返回空字符串
     */
    public static String toString(JsonNode node) {
        if (node == null || node.isNull() || StringUtils.isBlank(node.asText().trim()))
            return "";
        return node.asText().trim();
    }

    /**
     * 解析JsonNode中的String值并去除空格，解析不成功抛出异常
     */
    public static String toStringReq(JsonNode node, String message) throws ValidationException {
        if (node == null || node.isNull() || StringUtils.isBlank(node.asText().trim()))
            throw new ValidationException(message);
        return node.asText().trim();
    }

    //-------------------------------------------------------------------------

    /**
     * 解析JsonNode中的int值，解析不成功时返回默认值
     */
    public static int toInt(JsonNode node, int defaultValue) {
        if (node == null)
            return defaultValue;
        return DataUtils.toInt(node.asText(), defaultValue);
    }

    /**
     * 解析JsonNode中的Int值，解析不成功时返回null
     */
    public static Integer toInt(JsonNode node) {
        if (node == null)
            return null;
        return DataUtils.toInt(node.asText());
    }

    /**
     * 解析JsonNode中的int值，解析不成功时抛出异常
     */
    public static int toIntReq(JsonNode node, String message) throws ValidationException {
        if (node == null)
            throw new ValidationException(message);
        return DataUtils.toIntReq(node.asText(), message);
    }

    /**
     * 解析JsonNode中的Integer数组，解析不成功时返回空集合
     */
    public static List<Integer> toIntList(JsonNode node) {
        List<Integer> l = new ArrayList<>();
        if (node == null)
            return l;
        for (JsonNode nodeType : node) {
            Integer i = JsonNodeUtils.toInt(nodeType);
            if (i != null)
                l.add(i);
        }
        return l;
    }

    //-------------------------------------------------------------------------

    /**
     * 解析JsonNode中的long值，解析不成功时返回默认值
     */
    public static long toLong(JsonNode node, long defaultValue) {
        if (node == null)
            return defaultValue;
        return DataUtils.toLong(node.asText(), defaultValue);
    }

    /**
     * 解析JsonNode中的Long值，解析不成功时返回null
     */
    public static Long toLong(JsonNode node) {
        if (node == null)
            return null;
        return DataUtils.toLong(node.asText());
    }

    /**
     * 解析JsonNode中的long值，解析不成功时抛出异常
     */
    public static long toLongReq(JsonNode node, String message) throws ValidationException {
        if (node == null)
            throw new ValidationException(message);
        return DataUtils.toLongReq(node.asText(), message);
    }

    /**
     * 解析JsonNode中的Long数组，解析不成功时返回空集合
     */
    public static List<Long> toLongList(JsonNode node) {
        List<Long> l = new ArrayList<>();
        if (node == null)
            return l;
        for (JsonNode nodeType : node) {
            Long v = JsonNodeUtils.toLong(nodeType);
            if (v != null)
                l.add(v);
        }
        return l;
    }

    /**
     * 解析JsonNode中的String数组，解析不成功时返回空集合
     */
    public static List<String> toStringList(JsonNode node) {
        List<String> l = new ArrayList<>();
        if (node == null)
            return l;
        for (JsonNode nodeType : node) {
            String v = JsonNodeUtils.toString(nodeType);
            if (v != null)
                l.add(v);
        }
        return l;
    }

    //-------------------------------------------------------------------------

    /**
     * 解析JsonNode中的boolean值，解析不成功时返回默认值
     */
    public static boolean toBoolean(JsonNode node, boolean defaultValue) {
        if (node == null)
            return defaultValue;
        return DataUtils.toBoolean(node.asText());
    }

    /**
     * 解析JsonNode中的Boolean值，解析不成功时返回null
     */
    public static Boolean toBoolean(JsonNode node) {
        if (node == null || "null".equals(node.asText()))
            return null;
        return DataUtils.toBoolean(node.asText());
    }

    /**
     * 解析JsonNode中的boolean值，解析不成功时抛出异常
     */
    public static boolean toBooleanReq(JsonNode node, String message) throws ValidationException {
        if (node == null)
            throw new ValidationException(message);
        return DataUtils.toBoolean(node.asText());
    }

    //-------------------------------------------------------------------------

    /**
     * 解析JsonNode中的BigDecimal值，解析不成功时返回默认值
     */
    public static BigDecimal toBigDecimal(JsonNode node, BigDecimal defaultValue) {
        if (node == null)
            return defaultValue;
        return DataUtils.toBigDecimal(node.asText());
    }

    /**
     * 解析JsonNode中的BigDecimal值，解析不成功时返回null
     */
    public static BigDecimal toBigDecimal(JsonNode node) {
        if (node == null)
            return null;
        return DataUtils.toBigDecimal(node.asText());
    }

    /**
     * 解析JsonNode中的BigDecimal值，解析不成功时抛出异常
     */
    public static BigDecimal toBigDecimalReq(JsonNode node, String message) throws ValidationException {
        if (node == null)
            throw new ValidationException(message);
        return DataUtils.toBigDecimalReq(node.asText(), message);
    }

    //-------------------------------------------------------------------------

    /**
     * 解析JsonNode中的Date值，格式为：yyyy-MM-dd，丢弃时间部分，解析不成功时返回默认值
     */
    public static Date toDate(JsonNode node, Date defaultValue) {
        if (node == null)
            return defaultValue;
        return DataUtils.toDate(node.asText());
    }

    /**
     * 解析JsonNode中的Date值，格式为：yyyy-MM-dd，丢弃时间部分，解析不成功时返回null
     */
    public static Date toDate(JsonNode node) {
        if (node == null)
            return null;
        return DataUtils.toDate(node.asText());
    }

    /**
     * 解析JsonNode中的Date值，格式为：yyyy-MM-dd，丢弃时间部分，解析不成功时抛出异常
     */
    public static Date toDateReq(JsonNode node, String message) throws ValidationException {
        if (node == null)
            throw new ValidationException(message);
        return DataUtils.toDateReq(node.asText(), message);
    }

    //-------------------------------------------------------------------------

    /**
     * 解析JsonNode中的DateTime值，格式为：yyyy-MM-dd HH:mm:ss，丢弃毫秒部分，解析不成功时返回默认值
     */
    public static Date toDateTime(JsonNode node, Date defaultValue) {
        if (node == null)
            return defaultValue;
        return DataUtils.toDateTime(node.asText());
    }

    /**
     * 解析JsonNode中的DateTime值，格式为：yyyy-MM-dd HH:mm:ss，丢弃毫秒部分，解析不成功时返回null
     */
    public static Date toDateTime(JsonNode node) {
        if (node == null)
            return null;
        return DataUtils.toDateTime(node.asText());
    }

    /**
     * 解析JsonNode中的DateTime值，格式为：yyyy-MM-dd HH:mm:ss，丢弃毫秒部分，解析不成功时抛出异常
     */
    public static Date toDateTimeReq(JsonNode node, String message) throws ValidationException {
        if (node == null)
            throw new ValidationException(message);
        return DataUtils.toDateTimeReq(node.asText(), message);
    }

    /**
     * java 对象转换为 JsonNode
     */
    public static JsonNode objectToJsonNode(Object data) {
        JsonNode jsonNode = MAPPER.valueToTree(data);
        return jsonNode;
    }

    /**
     * object转对象
     * @param data
     * @param type
     * @param <T>
     * @return
     */
    public static <T> T objectToEntity(Object data,Class<T> type){
        MAPPER.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        T t=MAPPER.convertValue(data,type);
        return t;
    }

}
