package com.infoepoch.pms.dispatchassistant.common.utils;

import com.infoepoch.pms.dispatchassistant.common.exception.ValidationException;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

;

/**
 * 数据处理工具类
 */
public class DataUtils {

    /**
     * 解析int，解析不成功时返回默认值
     */
    public static int toInt(String s, int defaultValue) {
        try {
            return Integer.parseInt(s);
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    /**
     * 解析Integer，解析不成功时返回null
     */
    public static Integer toInt(String s) {
        try {
            return Integer.parseInt(s);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    /**
     * 解析int，解析不成功时抛出异常
     */
    public static int toIntReq(String s, String message) throws ValidationException {
        try {
            return Integer.parseInt(s);
        } catch (NumberFormatException e) {
            throw new ValidationException(message);
        }
    }

    //-------------------------------------------------------------------------

    /**
     * 解析long，解析不成功时返回默认值
     */
    public static long toLong(String s, long defaultValue) {
        try {
            return Long.parseLong(s);
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    /**
     * 解析Long，解析不成功时返回null
     */
    public static Long toLong(String s) {
        try {
            return Long.parseLong(s);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    /**
     * 解析long，解析不成功时抛出异常
     */
    public static long toLongReq(String s, String message) throws ValidationException {
        try {
            return Long.parseLong(s);
        } catch (NumberFormatException e) {
            throw new ValidationException(message);
        }
    }

    //-------------------------------------------------------------------------

    /**
     * 解析boolean，内部调用Boolean.parseBoolean方法
     * Boolean.parseBoolean兼容性极强，非true字符串都会解析成false，
     * 永远不会解析失败，故只保留一个方法
     * 详细测试用例可参考DataUtilsTest.testBoolean方法
     */
    public static boolean toBoolean(String s) {
        return Boolean.parseBoolean(s);
    }

    //-------------------------------------------------------------------------

    /**
     * 解析BigDecimal，解析不成功时返回默认值
     */
    public static BigDecimal toBigDecimal(String s, BigDecimal defaultValue) {
        if (StringUtils.isBlank(s))
            return defaultValue;
        try {
            return new BigDecimal(s);
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    /**
     * 解析BigDecimal，解析不成功时返回null
     */
    public static BigDecimal toBigDecimal(String s) {
        if (StringUtils.isBlank(s))
            return null;
        try {
            return new BigDecimal(s);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    /**
     * 解析BigDecimal，解析不成功时抛出异常
     */
    public static BigDecimal toBigDecimalReq(String s, String message) throws ValidationException {
        if (StringUtils.isBlank(s))
            throw new ValidationException(message);
        try {
            return new BigDecimal(s);
        } catch (NumberFormatException e) {
            throw new ValidationException(message);
        }
    }

    //-------------------------------------------------------------------------

    /**
     * 解析Date，格式为：yyyy-MM-dd，丢弃时间部分，解析不成功时返回默认值
     */
    public static Date toDate(String s, Date defaultValue) {
        try {
            SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd");
            return f.parse(s);
        } catch (ParseException e) {
            return defaultValue;
        }
    }

    /**
     * 解析Date，格式为：yyyy-MM-dd，丢弃时间部分，解析不成功时返回null
     */
    public static Date toDate(String s) {
        try {
            SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd");
            return f.parse(s);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 解析Date，格式为：yyyy-MM-dd，丢弃时间部分，解析不成功时抛出异常
     */
    public static Date toDateReq(String s, String message) throws ValidationException {
        try {
            SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd");
            return f.parse(s);
        } catch (ParseException e) {
            throw new ValidationException(message);
        }
    }

    //-------------------------------------------------------------------------

    /**
     * 解析DateTime，格式为：yyyy-MM-dd HH:mm:ss，丢弃毫秒部分，解析不成功时返回默认值
     */
    public static Date toDateTime(String s, Date defaultValue) {
        try {
            SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            return f.parse(s);
        } catch (ParseException e) {
            return defaultValue;
        }
    }

    /**
     * 解析DateTime，格式为：yyyy-MM-dd HH:mm:ss，丢弃毫秒部分，解析不成功时返回null
     */
    public static Date toDateTime(String s) {
        try {
            SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            return f.parse(s);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 解析DateTime，格式为：yyyy-MM-dd HH:mm:ss，丢弃毫秒部分，解析不成功时抛出异常
     */
    public static Date toDateTimeReq(String s, String message) throws ValidationException {
        try {
            SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            return f.parse(s);
        } catch (ParseException e) {
            throw new ValidationException(message);
        }
    }

}
