package com.infoepoch.pms.dispatchassistant.controller.basic;

import com.infoepoch.pms.dispatchassistant.common.component.Result;
import com.infoepoch.pms.dispatchassistant.common.exception.ValidationException;
import com.infoepoch.pms.dispatchassistant.domain.basic.todo.*;
import com.infoepoch.pms.dispatchassistant.domain.oa.OaService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Date;

/**
 *
 */
@RestController
@RequestMapping("/api/todo")
public class TodoController {

    @Autowired
    private OaService oaService;
    @Autowired
    private TodoService todoService;

    @Autowired
    private ITodoRepository todoRepository;

    /**
     * 根据条件查询列表（可分页）
     **/
    @PostMapping("/list")
    public Result queryList(@RequestBody TodoCriteria criteria) {
        return Result.successData(todoService.queryList(criteria));
    }

    /**
     *
     * @param todoId
     * @return
     */
    @PostMapping("/read/{todoId}")
    public Result read(@PathVariable String  todoId) {
        todoService.read(todoId);
        return Result.success();
    }

    /**
     * 根据id查询
     **/
    @GetMapping("get-by-id")
    public Result queryById(String id) {
        if (StringUtils.isBlank(id))
            throw new ValidationException("唯一标识不能为空");
        Todo entity = todoRepository.selectById(id);
        return Result.successData(entity);
    }

    /**
     * 保存
     **/
    @PostMapping("save")
    public Result save(@RequestBody Todo entity) {
        entity.verify();
        if(StringUtils.isBlank(entity.getId())) {
            entity.create();
            entity.setSendTime(new Date());
            todoRepository.insert(entity);
        } else {
            Todo old = todoRepository.selectById(entity.getId());
            if(old == null) {
                throw new ValidationException("未查询到可修改待办");
            }
            if(entity.getActionTime() == null && old.getActionTime() != null) {
                entity.setSyncStatus(0);
                if(TodoType.DONE.equals(old.getType())) {
                    entity.setType(TodoType.TODO);
                }
                if(TodoType.READ.equals(old.getType())) {
                    entity.setType(TodoType.TO_READ);
                }
            } else if(entity.getActionTime() != null && old.getActionTime() == null) {
                entity.setSyncStatus(0);
                if(TodoType.TODO.equals(old.getType())) {
                    entity.setType(TodoType.DONE);
                }
                if(TodoType.TO_READ.equals(old.getType())) {
                    entity.setType(TodoType.READ);
                }
            }
            old.modify(
                    entity.getDocumentType(),
                    entity.getDomainId(),
                    entity.getStage(),
                    entity.getDrafter(),
                    entity.getDrafterName(),
                    entity.getType(),
                    entity.getLevel(),
                    entity.getTitle(),
                    entity.getUrl(),
                    entity.getSender(),
                    entity.getSenderName(),
                    entity.getReceiver(),
                    entity.getReceiverName(),
                    entity.getActionTime(),
                    entity.getAdditionalData(),
                    entity.getSyncStatus()
            );
            todoRepository.update(old);
        }
        return Result.successData(entity);
    }

    /**
     * 根据ID删除待办
     *
     * @param id
     * @return
     */
    @GetMapping("/remove/{id}")
    public Result remove(@PathVariable("id") String id) {
        Todo todo = todoRepository.selectById(id);
        if (todo == null) {
            throw new ValidationException("无效的待办ID");
        }
        todoRepository.deleteById(id);
        return Result.success();
    }

    /**
     * 同步单个待办
     *
     * @param id
     * @return
     */
    @GetMapping("/sync-todo/{id}")
    public Result syncTodo(@PathVariable("id") String id) {
        Todo todo = todoRepository.selectById(id);
        if (todo == null) {
            return Result.error(-1, "无效的待办ID");
        }
        oaService.syncTodo(todo);
        return Result.success("待办同步成功");
    }

}
