package com.infoepoch.pms.dispatchassistant.common.component;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.BoundSetOperations;
import org.springframework.data.redis.core.HashOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.TimeUnit;

@Component
public class RedisTool {

    @Autowired
    private RedisTemplate redisTemplate;

    /**
     * 缓存基本对象：Integer、String、实体类等
     *
     * @param key   缓存KEY
     * @param value 缓存内容
     * @param <T>
     */
    public <T> void put(String key, T value) {
        redisTemplate.opsForValue().set(key, value);
    }

    /**
     * 缓存基本对象：Integer、String、实体类等
     *
     * @param key      缓存KEY
     * @param value    缓存内容
     * @param timeout  失效时间
     * @param timeUnit 失效时间单位
     * @param <T>
     */
    public <T> void put(String key, T value, long timeout, TimeUnit timeUnit) {
        redisTemplate.opsForValue().set(key, value, timeout, timeUnit);
    }

    /**
     * 设置缓存有效时间
     *
     * @param key     缓存KEY
     * @param timeout 失效时间（单位：秒）
     * @return
     */
    public boolean expire(String key, long timeout) {
        return expire(key, timeout, TimeUnit.SECONDS);
    }

    /**
     * 设置缓存有效时间
     *
     * @param key      缓存KEY
     * @param timeout  失效时间
     * @param timeUnit 失效时间单位
     * @return
     */
    public boolean expire(String key, long timeout, TimeUnit timeUnit) {
        return redisTemplate.expire(key, timeout, timeUnit);
    }

    /**
     * 获取缓存
     *
     * @param key 缓存KEY
     * @param <T>
     * @return
     */
    public <T> T get(String key) {
        ValueOperations<String, T> operations = redisTemplate.opsForValue();
        return operations.get(key);
    }

    /**
     * 删除缓存
     *
     * @param key 缓存KEY
     * @return
     */
    public boolean remove(String key) {
        return redisTemplate.delete(key);
    }

    /**
     * 删除集合对象
     *
     * @param collection
     * @return
     */
    public long removeCollection(Collection collection) {
        return redisTemplate.delete(collection);
    }

    /**
     * 设置LIST类型缓存数据
     *
     * @param key      缓存KEY
     * @param dataList LIST对象
     * @param <T>
     * @return
     */
    public <T> long putList(String key, List<T> dataList) {
        Long count = redisTemplate.opsForList().rightPushAll(key, dataList);
        return count == null ? 0 : count;
    }

    /**
     * 获取缓存的LIST对象
     *
     * @param key 缓存KEY
     * @param <T>
     * @return
     */
    public <T> List<T> getList(String key) {
        return redisTemplate.opsForList().range(key, 0, -1);
    }

    /**
     * 缓存SET
     *
     * @param key 缓存KEY
     * @param set SET对象
     * @param <T>
     * @return
     */
    public <T> BoundSetOperations<String, T> putSet(String key, Set<T> set) {
        BoundSetOperations<String, T> setOperations = redisTemplate.boundSetOps(key);
        Iterator<T> it = set.iterator();
        while (it.hasNext()) {
            setOperations.add(it.next());
        }
        return setOperations;
    }

    /**
     * 获取缓存SET
     *
     * @param key 缓存KEY
     * @param <T>
     * @return
     */
    public <T> Set<T> getSet(String key) {
        return redisTemplate.opsForSet().members(key);
    }

    /**
     * 缓存值到MAP中
     *
     * @param key
     * @param mapKey
     * @param value
     * @param <T>
     */
    public <T> void putMap(String key, String mapKey, T value) {
        if (value != null)
            redisTemplate.opsForHash().put(key, mapKey, value);
    }

    /**
     * 缓存MAP
     *
     * @param key 缓存KEY
     * @param map MAP对象
     * @param <T>
     */
    public <T> void putMap(String key, Map<String, T> map) {
        if (map != null)
            redisTemplate.opsForHash().putAll(key, map);
    }

    /**
     * 获取MAP中的值
     *
     * @param key    缓存KEY
     * @param mapKey MAP的key
     * @param <T>
     * @return
     */
    public <T> T getMapValue(String key, String mapKey) {
        HashOperations<String, String, T> operations = redisTemplate.opsForHash();
        return operations.get(key, mapKey);
    }

    /**
     * 获取多个MAP中的值
     *
     * @param key     缓存KEY
     * @param mapKeys MAP的key集合
     * @param <T>
     * @return
     */
    public <T> List<T> getMultiMapValue(String key, Collection<Object> mapKeys) {
        return redisTemplate.opsForHash().multiGet(key, mapKeys);
    }

    /**
     *
     * @param key
     * @param mapKey
     * @return
     */
    public boolean removeMapValue(String key, String mapKey) {
        return redisTemplate.opsForHash().delete(key, mapKey) > 0;
    }

    /**
     * 获取缓存的KEY列表
     *
     * @param pattern 正则表达式
     * @return
     */
    public Collection<String> keys(String pattern) {
        return redisTemplate.keys(pattern);
    }

}
