package com.infoepoch.pms.dispatchassistant.common.configuration;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.infoepoch.pms.dispatchassistant.common.enums.EnumBase;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.boot.autoconfigure.jackson.Jackson2ObjectMapperBuilderCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Optional;

/**
 * Jackson全局配置
 */
@Configuration
public class JacksonConfiguration {

    /**
     * 配置Spring MVC的Json转换器
     */
    @Bean
    public HttpMessageConverters jacksonConverters() {
        Jackson2ObjectMapperBuilder builder = new Jackson2ObjectMapperBuilder()
                .indentOutput(true)                                                 //输出时自动缩进
                .dateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"))    //日期时间格式化
                .serializerByType(Long.TYPE, new ToStringSerializer())              //long转String
                .serializerByType(Long.class, new ToStringSerializer())             //Long转String
                .serializerByType(EnumBase.class, new EnumSerializer())             //转换枚举
                ;
        ObjectMapper objectMapper = builder.build();
        objectMapper.setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);
        return new HttpMessageConverters(new MappingJackson2HttpMessageConverter(objectMapper));
    }

    /**
     * 配置@Autowired的ObjectMapper对象
     */
    @Bean
    public Jackson2ObjectMapperBuilderCustomizer jacksonCustomizer() {
        return builder -> {
            builder.indentOutput(true);
            builder.dateFormat(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
            builder.serializerByType(Long.TYPE, new ToStringSerializer());
            builder.serializerByType(Long.class, new ToStringSerializer());
            builder.serializerByType(EnumBase.class, new EnumSerializer());
        };
    }

    /**
     * 枚举序列化类
     */
    public class EnumSerializer extends JsonSerializer<EnumBase> {
        @Override
        public void serialize(EnumBase value, JsonGenerator jgen, SerializerProvider provider)
                throws IOException {
            if (value == null) {
                jgen.writeString("");
                return;
            }
            Optional<EnumBase> data = Optional.of(value);
            jgen.writeObject(data.get().getValue());
        }
    }

    /**
     * null值序列化类
     */
    public class NullDeserializer extends JsonDeserializer<Object> {
        @Override
        public Object deserialize(JsonParser jsonParser, DeserializationContext deserializationContext) throws IOException {
            return "";
        }
    }

}


